/*
 * Decompiled with CFR 0.152.
 */
package org.eclipse.jetty.util;

import java.io.File;
import java.io.IOException;
import java.lang.reflect.Field;
import java.nio.file.ClosedWatchServiceException;
import java.nio.file.FileSystems;
import java.nio.file.FileVisitResult;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.PathMatcher;
import java.nio.file.SimpleFileVisitor;
import java.nio.file.StandardWatchEventKinds;
import java.nio.file.WatchEvent;
import java.nio.file.WatchKey;
import java.nio.file.WatchService;
import java.nio.file.attribute.BasicFileAttributes;
import java.util.ArrayList;
import java.util.Collections;
import java.util.EventListener;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.concurrent.CopyOnWriteArrayList;
import java.util.concurrent.TimeUnit;
import org.eclipse.jetty.util.component.AbstractLifeCycle;
import org.eclipse.jetty.util.log.Log;
import org.eclipse.jetty.util.log.Logger;

public class PathWatcher
extends AbstractLifeCycle
implements Runnable {
    private static final boolean IS_WINDOWS;
    private static final Logger LOG;
    private static final Logger NOISY_LOG;
    private static final WatchEvent.Kind<?>[] WATCH_EVENT_KINDS;
    private WatchService watchService;
    private WatchEvent.Modifier[] watchModifiers;
    private boolean nativeWatchService;
    private Map<WatchKey, Config> keys = new HashMap<WatchKey, Config>();
    private List<EventListener> listeners = new CopyOnWriteArrayList<EventListener>();
    private List<Config> configs = new ArrayList<Config>();
    private long updateQuietTimeDuration = 1000L;
    private TimeUnit updateQuietTimeUnit = TimeUnit.MILLISECONDS;
    private Thread thread;
    private boolean _notifyExistingOnStart = true;
    private Map<Path, PathPendingEvents> pendingEvents = new LinkedHashMap<Path, PathPendingEvents>();

    protected static <T> WatchEvent<T> cast(WatchEvent<?> watchEvent) {
        return watchEvent;
    }

    public void watch(Path path) {
        Path path2 = path;
        if (!path2.isAbsolute()) {
            path2 = path.toAbsolutePath();
        }
        Config config = null;
        Path path3 = path2.getParent();
        for (Config config2 : this.configs) {
            if (!config2.getPath().equals(path3)) continue;
            config = config2;
            break;
        }
        if (config == null) {
            config = new Config(path2.getParent());
            config.addIncludeGlobRelative("");
            config.addIncludeGlobRelative(path.getFileName().toString());
            this.watch(config);
        } else {
            config.addIncludeGlobRelative(path.getFileName().toString());
        }
    }

    public void watch(Config config) {
        this.configs.add(config);
    }

    protected void prepareConfig(Config config) {
        if (LOG.isDebugEnabled()) {
            LOG.debug("Watching directory {}", config);
        }
        Files.walkFileTree(config.getPath(), new DepthLimitedFileVisitor(this, config));
    }

    public void addListener(EventListener eventListener) {
        this.listeners.add(eventListener);
    }

    private void appendConfigId(StringBuilder stringBuilder) {
        ArrayList<Path> arrayList = new ArrayList<Path>();
        for (Config config : this.keys.values()) {
            arrayList.add(config.dir);
        }
        Collections.sort(arrayList);
        stringBuilder.append("[");
        if (arrayList.size() > 0) {
            stringBuilder.append(arrayList.get(0));
            if (arrayList.size() > 1) {
                stringBuilder.append(" (+").append(arrayList.size() - 1).append(")");
            }
        } else {
            stringBuilder.append("<null>");
        }
        stringBuilder.append("]");
    }

    @Override
    protected void doStart() {
        this.createWatchService();
        this.setUpdateQuietTime(this.getUpdateQuietTimeMillis(), TimeUnit.MILLISECONDS);
        for (Config config : this.configs) {
            this.prepareConfig(config);
        }
        StringBuilder stringBuilder = new StringBuilder();
        stringBuilder.append("PathWatcher-Thread");
        this.appendConfigId(stringBuilder);
        this.thread = new Thread((Runnable)this, stringBuilder.toString());
        this.thread.setDaemon(true);
        this.thread.start();
        super.doStart();
    }

    @Override
    protected void doStop() {
        if (this.watchService != null) {
            this.watchService.close();
        }
        this.watchService = null;
        this.thread = null;
        this.keys.clear();
        this.pendingEvents.clear();
        super.doStop();
    }

    public void reset() {
        if (!this.isStopped()) {
            throw new IllegalStateException("PathWatcher must be stopped before reset.");
        }
        this.configs.clear();
        this.listeners.clear();
    }

    private void createWatchService() {
        this.watchService = FileSystems.getDefault().newWatchService();
        WatchEvent.Modifier[] modifierArray = null;
        boolean bl = true;
        try {
            ClassLoader classLoader = Thread.currentThread().getContextClassLoader();
            Class<?> clazz = Class.forName("sun.nio.fs.PollingWatchService", false, classLoader);
            if (clazz.isAssignableFrom(this.watchService.getClass())) {
                bl = false;
                LOG.info("Using Non-Native Java {}", clazz.getName());
                Class<?> clazz2 = Class.forName("com.sun.nio.file.SensitivityWatchEventModifier");
                Field field = clazz2.getField("HIGH");
                modifierArray = new WatchEvent.Modifier[]{(WatchEvent.Modifier)field.get(clazz2)};
            }
        }
        catch (Throwable throwable) {
            LOG.ignore(throwable);
        }
        this.watchModifiers = modifierArray;
        this.nativeWatchService = bl;
    }

    protected boolean isNotifiable() {
        return this.isStarted() || !this.isStarted() && this.isNotifyExistingOnStart();
    }

    public Iterator<EventListener> getListeners() {
        return this.listeners.iterator();
    }

    public long getUpdateQuietTimeMillis() {
        return TimeUnit.MILLISECONDS.convert(this.updateQuietTimeDuration, this.updateQuietTimeUnit);
    }

    protected void notifyOnPathWatchEvents(List<PathWatchEvent> list) {
        if (list == null || list.isEmpty()) {
            return;
        }
        for (EventListener eventListener : this.listeners) {
            if (eventListener instanceof EventListListener) {
                try {
                    ((EventListListener)eventListener).onPathWatchEvents(list);
                }
                catch (Throwable throwable) {
                    LOG.warn(throwable);
                }
                continue;
            }
            Listener listener = (Listener)eventListener;
            for (PathWatchEvent pathWatchEvent : list) {
                try {
                    listener.onPathWatchEvent(pathWatchEvent);
                }
                catch (Throwable throwable) {
                    LOG.warn(throwable);
                }
            }
        }
    }

    protected void register(Path path, Config config) {
        LOG.debug("Registering watch on {}", path);
        if (this.watchModifiers != null) {
            WatchKey watchKey = path.register(this.watchService, WATCH_EVENT_KINDS, this.watchModifiers);
            this.keys.put(watchKey, config.asSubConfig(path));
        } else {
            WatchKey watchKey = path.register(this.watchService, WATCH_EVENT_KINDS);
            this.keys.put(watchKey, config.asSubConfig(path));
        }
    }

    public boolean removeListener(Listener listener) {
        return this.listeners.remove(listener);
    }

    @Override
    public void run() {
        ArrayList<PathWatchEvent> arrayList = new ArrayList<PathWatchEvent>();
        if (LOG.isDebugEnabled()) {
            LOG.debug("Starting java.nio file watching with {}", this.watchService);
        }
        while (this.watchService != null && this.thread == Thread.currentThread()) {
            Object object;
            WatchKey watchKey = null;
            try {
                if (this.pendingEvents.isEmpty()) {
                    if (NOISY_LOG.isDebugEnabled()) {
                        NOISY_LOG.debug("Waiting for take()", new Object[0]);
                    }
                    watchKey = this.watchService.take();
                } else {
                    if (NOISY_LOG.isDebugEnabled()) {
                        NOISY_LOG.debug("Waiting for poll({}, {})", new Object[]{this.updateQuietTimeDuration, this.updateQuietTimeUnit});
                    }
                    if ((watchKey = this.watchService.poll(this.updateQuietTimeDuration, this.updateQuietTimeUnit)) == null) {
                        long l = System.currentTimeMillis();
                        for (Path path : new HashSet<Path>(this.pendingEvents.keySet())) {
                            object = this.pendingEvents.get(path);
                            if (!((PathPendingEvents)object).isQuiet(l, this.updateQuietTimeDuration, this.updateQuietTimeUnit)) continue;
                            for (PathWatchEvent pathWatchEvent : ((PathPendingEvents)object).getEvents()) {
                                arrayList.add(pathWatchEvent);
                            }
                            this.pendingEvents.remove(path);
                        }
                    }
                }
            }
            catch (ClosedWatchServiceException closedWatchServiceException) {
                return;
            }
            catch (InterruptedException interruptedException) {
                if (this.isRunning()) {
                    LOG.warn(interruptedException);
                } else {
                    LOG.ignore(interruptedException);
                }
                return;
            }
            if (watchKey != null) {
                Config config = this.keys.get(watchKey);
                if (config == null) {
                    if (!LOG.isDebugEnabled()) continue;
                    LOG.debug("WatchKey not recognized: {}", watchKey);
                    continue;
                }
                for (WatchEvent<?> watchEvent : watchKey.pollEvents()) {
                    WatchEvent.Kind<?> kind = watchEvent.kind();
                    object = PathWatcher.cast(watchEvent);
                    Path path = (Path)object.context();
                    Path path2 = config.dir.resolve(path);
                    if (kind == StandardWatchEventKinds.ENTRY_CREATE) {
                        if (Files.isDirectory(path2, LinkOption.NOFOLLOW_LINKS)) {
                            try {
                                this.prepareConfig(config.asSubConfig(path2));
                            }
                            catch (IOException iOException) {
                                LOG.warn(iOException);
                            }
                            continue;
                        }
                        if (!config.matches(path2)) continue;
                        this.addToPendingList(path2, new PathWatchEvent(path2, (WatchEvent<Path>)object));
                        continue;
                    }
                    if (!config.matches(path2)) continue;
                    this.addToPendingList(path2, new PathWatchEvent(path2, (WatchEvent<Path>)object));
                }
            }
            this.notifyOnPathWatchEvents(arrayList);
            arrayList.clear();
            if (watchKey == null || watchKey.reset()) continue;
            this.keys.remove(watchKey);
            if (!this.keys.isEmpty()) continue;
            return;
        }
    }

    public void addToPendingList(Path path, PathWatchEvent pathWatchEvent) {
        PathPendingEvents pathPendingEvents = this.pendingEvents.get(path);
        if (pathPendingEvents == null) {
            this.pendingEvents.put(path, new PathPendingEvents(path, pathWatchEvent));
        } else {
            pathPendingEvents.addEvent(pathWatchEvent);
        }
    }

    public void setNotifyExistingOnStart(boolean bl) {
        this._notifyExistingOnStart = bl;
    }

    public boolean isNotifyExistingOnStart() {
        return this._notifyExistingOnStart;
    }

    public void setUpdateQuietTime(long l, TimeUnit timeUnit) {
        long l2 = timeUnit.toMillis(l);
        if (this.watchService != null && !this.nativeWatchService && l2 < 5000L) {
            LOG.warn("Quiet Time is too low for non-native WatchService [{}]: {} < 5000 ms (defaulting to 5000 ms)", this.watchService.getClass().getName(), l2);
            this.updateQuietTimeDuration = 5000L;
            this.updateQuietTimeUnit = TimeUnit.MILLISECONDS;
            return;
        }
        if (IS_WINDOWS && l2 < 1000L) {
            LOG.warn("Quiet Time is too low for Microsoft Windows: {} < 1000 ms (defaulting to 1000 ms)", l2);
            this.updateQuietTimeDuration = 1000L;
            this.updateQuietTimeUnit = TimeUnit.MILLISECONDS;
            return;
        }
        this.updateQuietTimeDuration = l;
        this.updateQuietTimeUnit = timeUnit;
    }

    public String toString() {
        StringBuilder stringBuilder = new StringBuilder(this.getClass().getName());
        this.appendConfigId(stringBuilder);
        return stringBuilder.toString();
    }

    static {
        String string = System.getProperty("os.name");
        if (string == null) {
            IS_WINDOWS = false;
        } else {
            String string2 = string.toLowerCase(Locale.ENGLISH);
            IS_WINDOWS = string2.contains("windows");
        }
        LOG = Log.getLogger(PathWatcher.class);
        NOISY_LOG = Log.getLogger(PathWatcher.class.getName() + ".Noisy");
        WATCH_EVENT_KINDS = new WatchEvent.Kind[]{StandardWatchEventKinds.ENTRY_CREATE, StandardWatchEventKinds.ENTRY_DELETE, StandardWatchEventKinds.ENTRY_MODIFY};
    }

    public static enum PathWatchEventType {
        ADDED,
        DELETED,
        MODIFIED,
        UNKNOWN;

    }

    public static class PathPendingEvents {
        private Path _path;
        private List<PathWatchEvent> _events;
        private long _timestamp;
        private long _lastFileSize = -1L;

        public PathPendingEvents(Path path) {
            this._path = path;
        }

        public PathPendingEvents(Path path, PathWatchEvent pathWatchEvent) {
            this(path);
            this.addEvent(pathWatchEvent);
        }

        public void addEvent(PathWatchEvent pathWatchEvent) {
            long l;
            this._timestamp = l = System.currentTimeMillis();
            if (this._events == null) {
                this._events = new ArrayList<PathWatchEvent>();
                this._events.add(pathWatchEvent);
            } else {
                PathWatchEvent pathWatchEvent2 = null;
                for (PathWatchEvent pathWatchEvent3 : this._events) {
                    if (pathWatchEvent3.getType() != pathWatchEvent.getType()) continue;
                    pathWatchEvent2 = pathWatchEvent3;
                    break;
                }
                if (pathWatchEvent2 == null) {
                    this._events.add(pathWatchEvent);
                } else {
                    pathWatchEvent2.incrementCount(pathWatchEvent.getCount());
                }
            }
        }

        public List<PathWatchEvent> getEvents() {
            return this._events;
        }

        public long getTimestamp() {
            return this._timestamp;
        }

        public boolean isQuiet(long l, long l2, TimeUnit timeUnit) {
            long l3 = this._timestamp + timeUnit.toMillis(l2);
            this._timestamp = l;
            long l4 = this._path.toFile().length();
            boolean bl = this._lastFileSize != l4;
            this._lastFileSize = l4;
            return l > l3 && !bl;
        }
    }

    public static class PathWatchEvent {
        private final Path path;
        private final PathWatchEventType type;
        private int count = 0;

        public PathWatchEvent(Path path, PathWatchEventType pathWatchEventType) {
            this.path = path;
            this.count = 1;
            this.type = pathWatchEventType;
        }

        public PathWatchEvent(Path path, WatchEvent<Path> watchEvent) {
            this.path = path;
            this.count = watchEvent.count();
            this.type = watchEvent.kind() == StandardWatchEventKinds.ENTRY_CREATE ? PathWatchEventType.ADDED : (watchEvent.kind() == StandardWatchEventKinds.ENTRY_DELETE ? PathWatchEventType.DELETED : (watchEvent.kind() == StandardWatchEventKinds.ENTRY_MODIFY ? PathWatchEventType.MODIFIED : PathWatchEventType.UNKNOWN));
        }

        public boolean equals(Object object) {
            if (this == object) {
                return true;
            }
            if (object == null) {
                return false;
            }
            if (this.getClass() != object.getClass()) {
                return false;
            }
            PathWatchEvent pathWatchEvent = (PathWatchEvent)object;
            if (this.path == null ? pathWatchEvent.path != null : !this.path.equals(pathWatchEvent.path)) {
                return false;
            }
            return this.type == pathWatchEvent.type;
        }

        public Path getPath() {
            return this.path;
        }

        public PathWatchEventType getType() {
            return this.type;
        }

        public void incrementCount(int n) {
            this.count += n;
        }

        public int getCount() {
            return this.count;
        }

        public int hashCode() {
            int n = 31;
            int n2 = 1;
            n2 = 31 * n2 + (this.path == null ? 0 : this.path.hashCode());
            n2 = 31 * n2 + (this.type == null ? 0 : this.type.hashCode());
            return n2;
        }

        public String toString() {
            return String.format("PathWatchEvent[%s|%s]", new Object[]{this.type, this.path});
        }
    }

    public static interface EventListListener
    extends EventListener {
        public void onPathWatchEvents(List<PathWatchEvent> var1);
    }

    public static interface Listener
    extends EventListener {
        public void onPathWatchEvent(PathWatchEvent var1);
    }

    public static class DepthLimitedFileVisitor
    extends SimpleFileVisitor<Path> {
        private Config base;
        private PathWatcher watcher;

        public DepthLimitedFileVisitor(PathWatcher pathWatcher, Config config) {
            this.base = config;
            this.watcher = pathWatcher;
        }

        @Override
        public FileVisitResult preVisitDirectory(Path path, BasicFileAttributes basicFileAttributes) {
            if (!this.base.isExcluded(path)) {
                if (this.base.isIncluded(path) && this.watcher.isNotifiable()) {
                    PathWatchEvent pathWatchEvent = new PathWatchEvent(path, PathWatchEventType.ADDED);
                    if (LOG.isDebugEnabled()) {
                        LOG.debug("Pending {}", pathWatchEvent);
                    }
                    this.watcher.addToPendingList(path, pathWatchEvent);
                }
                if (this.base.getPath().equals(path) && (this.base.isRecurseDepthUnlimited() || this.base.getRecurseDepth() >= 0) || this.base.shouldRecurseDirectory(path)) {
                    this.watcher.register(path, this.base);
                }
            }
            if (this.base.getPath().equals(path) && (this.base.isRecurseDepthUnlimited() || this.base.getRecurseDepth() >= 0) || this.base.shouldRecurseDirectory(path)) {
                return FileVisitResult.CONTINUE;
            }
            return FileVisitResult.SKIP_SUBTREE;
        }

        @Override
        public FileVisitResult visitFile(Path path, BasicFileAttributes basicFileAttributes) {
            if (this.base.matches(path) && this.watcher.isNotifiable()) {
                PathWatchEvent pathWatchEvent = new PathWatchEvent(path, PathWatchEventType.ADDED);
                if (LOG.isDebugEnabled()) {
                    LOG.debug("Pending {}", pathWatchEvent);
                }
                this.watcher.addToPendingList(path, pathWatchEvent);
            }
            return FileVisitResult.CONTINUE;
        }
    }

    public static class Config {
        public static final int UNLIMITED_DEPTH = -9999;
        private static final String PATTERN_SEP;
        protected final Path dir;
        protected int recurseDepth = 0;
        protected List<PathMatcher> includes;
        protected List<PathMatcher> excludes;
        protected boolean excludeHidden = false;

        public Config(Path path) {
            this.dir = path;
            this.includes = new ArrayList<PathMatcher>();
            this.excludes = new ArrayList<PathMatcher>();
        }

        public void addExclude(PathMatcher pathMatcher) {
            this.excludes.add(pathMatcher);
        }

        public void addExclude(String string) {
            if (LOG.isDebugEnabled()) {
                LOG.debug("Adding exclude: [{}]", string);
            }
            this.addExclude(this.dir.getFileSystem().getPathMatcher(string));
        }

        public void addExcludeGlobRelative(String string) {
            this.addExclude(this.toGlobPattern(this.dir, string));
        }

        public void addExcludeHidden() {
            if (!this.excludeHidden) {
                if (LOG.isDebugEnabled()) {
                    LOG.debug("Adding hidden files and directories to exclusions", new Object[0]);
                }
                this.excludeHidden = true;
                this.addExclude("regex:^.*" + PATTERN_SEP + "\\..*$");
                this.addExclude("regex:^.*" + PATTERN_SEP + "\\..*" + PATTERN_SEP + ".*$");
            }
        }

        public void addExcludes(List<String> list) {
            for (String string : list) {
                this.addExclude(string);
            }
        }

        public void addInclude(PathMatcher pathMatcher) {
            this.includes.add(pathMatcher);
        }

        public void addInclude(String string) {
            if (LOG.isDebugEnabled()) {
                LOG.debug("Adding include: [{}]", string);
            }
            this.addInclude(this.dir.getFileSystem().getPathMatcher(string));
        }

        public void addIncludeGlobRelative(String string) {
            this.addInclude(this.toGlobPattern(this.dir, string));
        }

        public void addIncludes(List<String> list) {
            for (String string : list) {
                this.addInclude(string);
            }
        }

        public Config asSubConfig(Path path) {
            Config config = new Config(path);
            config.includes = this.includes;
            config.excludes = this.excludes;
            config.recurseDepth = path == this.dir ? this.recurseDepth : (this.recurseDepth == -9999 ? -9999 : this.recurseDepth - (path.getNameCount() - this.dir.getNameCount()));
            return config;
        }

        public int getRecurseDepth() {
            return this.recurseDepth;
        }

        public boolean isRecurseDepthUnlimited() {
            return this.recurseDepth == -9999;
        }

        public Path getPath() {
            return this.dir;
        }

        private boolean hasMatch(Path path, List<PathMatcher> list) {
            for (PathMatcher pathMatcher : list) {
                if (!pathMatcher.matches(path)) continue;
                return true;
            }
            return false;
        }

        public boolean isExcluded(Path path) {
            if (this.excludeHidden && Files.isHidden(path)) {
                if (NOISY_LOG.isDebugEnabled()) {
                    NOISY_LOG.debug("isExcluded [Hidden] on {}", path);
                }
                return true;
            }
            if (this.excludes.isEmpty()) {
                return false;
            }
            boolean bl = this.hasMatch(path, this.excludes);
            if (NOISY_LOG.isDebugEnabled()) {
                NOISY_LOG.debug("isExcluded [{}] on {}", bl, path);
            }
            return bl;
        }

        public boolean isIncluded(Path path) {
            if (this.includes.isEmpty()) {
                if (NOISY_LOG.isDebugEnabled()) {
                    NOISY_LOG.debug("isIncluded [All] on {}", path);
                }
                return true;
            }
            boolean bl = this.hasMatch(path, this.includes);
            if (NOISY_LOG.isDebugEnabled()) {
                NOISY_LOG.debug("isIncluded [{}] on {}", bl, path);
            }
            return bl;
        }

        public boolean matches(Path path) {
            try {
                return !this.isExcluded(path) && this.isIncluded(path);
            }
            catch (IOException iOException) {
                LOG.warn("Unable to match path: " + path, iOException);
                return false;
            }
        }

        public void setRecurseDepth(int n) {
            this.recurseDepth = n;
        }

        public boolean shouldRecurseDirectory(Path path) {
            if (!path.startsWith(this.dir)) {
                return false;
            }
            if (this.isRecurseDepthUnlimited()) {
                return true;
            }
            int n = this.dir.relativize(path).getNameCount();
            return n <= this.recurseDepth;
        }

        private String toGlobPattern(Path path, String string) {
            StringBuilder stringBuilder = new StringBuilder();
            stringBuilder.append("glob:");
            boolean bl = false;
            Path path2 = path.getRoot();
            if (path2 != null) {
                if (NOISY_LOG.isDebugEnabled()) {
                    NOISY_LOG.debug("Path: {} -> Root: {}", path, path2);
                }
                for (Object object : (Object)path2.toString().toCharArray()) {
                    if (object == 92) {
                        stringBuilder.append(PATTERN_SEP);
                        continue;
                    }
                    stringBuilder.append((char)object);
                }
            } else {
                bl = true;
            }
            for (Path path3 : path) {
                if (bl) {
                    stringBuilder.append(PATTERN_SEP);
                }
                stringBuilder.append(path3);
                bl = true;
            }
            if (string != null && string.length() > 0) {
                if (bl) {
                    stringBuilder.append(PATTERN_SEP);
                }
                for (Object object : (Object)string.toCharArray()) {
                    if (object == 47) {
                        stringBuilder.append(PATTERN_SEP);
                        continue;
                    }
                    stringBuilder.append((char)object);
                }
            }
            return stringBuilder.toString();
        }

        public String toString() {
            StringBuilder stringBuilder = new StringBuilder();
            stringBuilder.append(this.dir);
            if (this.recurseDepth > 0) {
                stringBuilder.append(" [depth=").append(this.recurseDepth).append("]");
            }
            return stringBuilder.toString();
        }

        static {
            String string = File.separator;
            if (File.separatorChar == '\\') {
                string = "\\\\";
            }
            PATTERN_SEP = string;
        }
    }
}

